#!/bin/sh

set -e

error() {
	echo "ERROR: $1" >&2
	exit 1
}

whoami | grep root > /dev/null && error "This script should not be run as root."

if [ "$#" -ne 1 ]; then
	echo "Adjust settings of a DAW / host application running in a Flatpak sandbox in order to allow Charlatan accessing your presets folder and work properly overall:"
	echo "- Add user permission to access desktop session bus (D-Bus)"
	echo "- Create symlink from sandboxed environment to your user preset directoy."
	echo
	echo "Syntax: flatpak_fix.sh <app-id>"
	echo
	echo "where app-id is the Flatpak application ID, such as org.ardour.Ardour."
	exit 0
fi

app="$1"
flatpak info "$app" > /dev/null
flatpak override -u --socket=session-bus "$app"

if [ -d "$HOST_XDG_DATA_HOME" ]; then
	echo "\$HOST_XDG_DATA_HOME has been set => no symlink will be created"
else
	DATA_DIR="${XDG_DATA_HOME:-$HOME/.local/share}"
	DATA_DIR="$DATA_DIR/charlatan"

	app_data_base_dir="$HOME/.var/app/$app/data"
	app_data_dir="$app_data_base_dir/charlatan"

	if [ -d "$app_data_dir" ]; then
		rmdir "$app_data_dir" >/dev/null 2>&1 || true
	fi

	if [ -L "$app_data_dir" ]; then
		rm "$app_data_dir" >/dev/null 2>&1 || true
	fi

	if [ -d "$app_data_dir" ]; then
		echo "WARNING: Cannot create symlink $app_data_dir, path already exists"
	else
		mkdir -p "$app_data_base_dir"
		echo "Creating symlink $app_data_dir -> $DATA_DIR"
		ln -s "$DATA_DIR" "$app_data_dir"
	fi
fi
